using System;

namespace HIPS.Web.Components.Cache
{
    public abstract class CacheProvider : ICacheProvider
    {
        /// <summary>
        /// Gets or sets the default sliding expiration.
        /// The sliding expiration indicates when a cache entry should be evicted if it has not been accessed in a given span of time.
        /// Default value is NoSlidingExpiration.
        /// </summary>
        public TimeSpan DefaultSlidingExpiration { get; set; }

        /// <summary>
        /// Gets or sets the default absolute expiration offset.
        /// The absolute expiration offset is applied to the current date/time to determine an absolute expiration.
        /// The absolute expiration indicates whether a cache entry should be evicted at a specified date and time.
        /// Default value is NoAbsoluteExpirationOffset.
        /// </summary>
        public TimeSpan DefaultAbsoluteExpirationOffset { get; set; }

        /// <summary>
        /// Used as a parameter to indicate caching should not use absolute expirations. This field is read-only.
        /// </summary>
        public static readonly TimeSpan NoAbsoluteExpirationOffset = TimeSpan.MaxValue;

        /// <summary>
        /// Used as a parameter to indicate caching should not use sliding expirations. This field is read-only.
        /// </summary>
        public static readonly TimeSpan NoSlidingExpiration = TimeSpan.Zero;


        /// <summary>
        /// Initializes a new instance of the <see cref="CacheProvider"/> class.
        /// </summary>
        /// <remarks>
        /// If using this constructor, method calls may need to explicitly specify an expiry if caching is desired.
        /// </remarks>
        protected CacheProvider()
        {
            DefaultSlidingExpiration = NoSlidingExpiration;
            DefaultAbsoluteExpirationOffset = NoAbsoluteExpirationOffset;
        }

        /// <summary>
        /// Initializes a new instance of the <see cref="CacheProvider"/> class.
        /// </summary>
        /// <param name="defaultAbsoluteExpirationOffset">The default absolute expiration offset, or null if not using absolute expiration.</param>
        /// <param name="defaultSlidingExpiration">The default sliding expiration, or null if not using default sliding expiration.</param>
        protected CacheProvider(TimeSpan? defaultAbsoluteExpirationOffset, TimeSpan? defaultSlidingExpiration = null)
        {
            DefaultSlidingExpiration = defaultSlidingExpiration ?? NoSlidingExpiration;
            DefaultAbsoluteExpirationOffset = defaultAbsoluteExpirationOffset ?? NoAbsoluteExpirationOffset;
        }

        /// <summary>
        /// Gets the value identified by the cache key from the cache.
        /// Returns null if there is no cached value for the specified key.
        /// </summary>
        /// <typeparam name="T">Type of the item to be retrieved from the cache.</typeparam>
        /// <param name="cacheKey">A unique identifier for the cache entry.</param>
        /// <returns>
        /// The cached value for the specified key, or null if no value found in the cache.
        /// </returns>
        public abstract T Get<T>(string cacheKey) where T : class;

        /// <summary>
        /// Stores the provided value against the identified cache key. 
        /// Uses default expiry settings.
        /// </summary>
        /// <typeparam name="T">Type of the item to be stored in the cache.</typeparam>
        /// <param name="cacheKey">A unique identifier for the cache entry.</param>
        /// <param name="item">The item value to set.</param>
        /// <param name="dependencyKeys">If supported, an optional set of keys that this entry is dependent on. If these keys do not exist in the cache, or are changed or removed, this entry is removed from the cache.</param>
        public void Set<T>(string cacheKey, T item, params string[] dependencyKeys) where T : class
        {
            Set(cacheKey, item, GetDefaultSlidingExpiration(), GetDefaultAbsoluteExpiration(), dependencyKeys);
        }

        /// <summary>
        /// Stores the provided value against the identified cache key.
        /// If the provided value is null the identified cache key is cleared.
        /// </summary>
        /// <typeparam name="T">Type of the item to be stored in the cache.</typeparam>
        /// <param name="cacheKey">A unique identifier for the cache entry.</param>
        /// <param name="item">The item value to set. If null is provided the key is cleared.</param>
        /// <param name="slidingExpiration">A value indicating when a cache entry should be evicted if it has not been accessed in a given span of time.</param>
        /// <param name="absoluteExpiration">A value indicating whether a cache entry should be evicted at a specified date and time.</param>
        /// <param name="dependencyKeys">If supported, an optional set of keys that this entry is dependent on. If these keys do not exist in the cache, or are changed or removed, this entry is removed from the cache.</param>
        public abstract void Set<T>(string cacheKey, T item, TimeSpan slidingExpiration, DateTimeOffset absoluteExpiration, params string[] dependencyKeys) where T : class;

        /// <summary>
        /// Returns a reference to an object that can be used as an applicable thread lock for this cache for the specified key.
        /// This is useful for functionality such as double-check locking. 
        ///  </summary>
        /// <param name="cacheKey">The cache key.</param>
        /// <returns>A cache-specific key-specific thread lock.</returns>
        public abstract object GetLock(string cacheKey);

        /// <summary>
        /// Gets the default sliding expiration for new cache values.
        /// The sliding expiration indicates when a cache entry should be evicted if it has not been accessed in a given span of time.
        /// </summary>
        public TimeSpan GetDefaultSlidingExpiration()
        {
            return DefaultSlidingExpiration;
        }

        /// <summary>
        /// Gets the default absolute expiration for new cache values.
        /// The absolute expiration indicates whether a cache entry should be evicted at a specified date and time.
        /// </summary>
        /// <returns></returns>
        public DateTimeOffset GetDefaultAbsoluteExpiration()
        {
            return DefaultAbsoluteExpirationOffset == NoAbsoluteExpirationOffset
                ? DateTimeOffset.MaxValue
                : DateTime.UtcNow.Add(DefaultAbsoluteExpirationOffset);
        }
    }
}